""" Example model 2

Generates predictions to single-choice syllogistic tasks by randomly sampling
from the set of choices. Sampling probabilities are dependent on previously
observed responses.

"""

import ccobra
import numpy as np

class AdaptModel(ccobra.CCobraModel):
    def __init__(self, name='AdaptModel'):
        """ Initializes the RandomModel by calling the parent-class constructor
        and passing information about the name as well as supported domains
        and response-types.

        Parameters
        ----------
        name : str
            Name of the model. Will be used as an identifier throughout the
            evaluation phase. Should be unique.

        """

        super(AdaptModel, self).__init__(
            name, ['syllogistic'], ['single-choice'])

        # Initialize the frequency dict which will be used for specifying the
        # prediction distribution. Frequencies are initialized to an epsilon
        # value to prevent a division by zero when transforming frequencies to
        # probabilities.
        self.frequencies = {}
        epsilon = 0.00001
        for response in ccobra.syllogistic.RESPONSES:
            self.frequencies[response] = epsilon

    def predict(self, item, **kwargs):
        """ Generates a prediction based on a given task item.

        Parameters
        ----------
        item : ccobra.data.Item
            Task item container. Holds information about the task, domain,
            response type and response choices.

        Returns
        -------
        list(str)
            Single-choice syllogistic response in list representation (e.g.,
            ['All', 'managers', 'clerks']).

        """

        # Instantiate the syllogistic helper object in order to conveniently
        # obtain the string-representations of the response choices.
        syl = ccobra.syllogistic.Syllogism(item)

        # Convert the observed frequencies to response probabilities
        probs = [self.frequencies[syl.encode_response(x)] for x in item.choices]
        probs = np.array(probs) / np.sum(probs)

        # Sample the prediction based on the observation probabilities
        return np.random.choice(item.choices, p=probs)

    def adapt(self, item, truth, **kwargs):
        """ Adapt the model to the individual currently being simulated.

        Parameters
        ----------
        item : ccobra.data.Item
            Task item container. Holds information about the task, domain,
            response type and response choices.

        truth : list(str)
            True response given by the human individual.

        """

        # Instantiate the syllogistic helper object in order to conveniently
        # obtain the string-representations of the response choices.
        syl = ccobra.syllogistic.Syllogism(item)
        enc_truth = syl.encode_response(truth)

        # Update the dictionary containing observed response frequencies
        self.frequencies[enc_truth] += 1

    def pre_train(self, dataset, **kwargs):
        """ Pre-trains the model based on given information about other
        individuals. Uses the adaption scheme in this case, but could also
        implement a more elaborate pre-training mechanism.

        Parameters
        ----------
        dataset : list(list(dict))
            Pre-training dataset of shape (n_individuals, n_tasks, ...) with
            n_individuals being the number of human reasoners found in the
            dataset and n_tasks denoting the number of tasks they responded
            to (i.e., 64 for full syllogistic experiments). For each task,
            the item as well as the response is given (plus additional
            information contained in the data).

        """

        # Iterate over subjects in the data and the tasks they responded to
        for subj_train_data in dataset:
            for seq_train_data in subj_train_data:
                # Adapt the model based on the human response
                self.adapt(seq_train_data['item'], seq_train_data['response'])
